package gov.va.med.domain.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Used to encapsulate a collection of DTO objects that
 * came in from a messaging request.  It contains:
 * <UL>
 * <LI>A list of DTOs built from a message with multipe entries. 
 * <LI>A Map of error lists.  These error lists are either associate with 
 * a DTO or are orphaned. <P>
 * <UL>
 * <LI>If associated with a DTO,  the key of is the errant DTO itself. <BR>
 * <LI>If orphaned, the key is either the HL7 Segemnt name 
 * in which they were found (using the pattern segmentName_ERRORS) <BR> 
 * or the MESSAGE_ERRORS if they are not specific to a Segment.  
 * </UL>
 * </UL>
 * @author Tom Cornils
 * @version $Id: DtoListPayload.java,v 1.13 2005/07/21 19:39:32 joel.goldberg Exp $
 * @since MHV 2.0 <br>03/17/2005
 */
public class DtoListPayload implements IPayload {
	public static final String MESSAGE_ERRORS = "MESSAGE_ERRORS";
    private List dtoList;
	private Map errors;
	private MessagingRequest request; 
	
    public boolean containsHealthRecord() {
        Iterator dtos = getDtoIterator();
        while (dtos.hasNext()) {
            if (dtos.next() instanceof HealthRecordDomainObject ) 
            { 
                return true;
            }
        }
        return false; 
    }
    /**
     * Selects only errors keyed by a DTO and returns a Set of these DTOs. 
     * An object is assumed to be a DTO is it is not a String.   
     * <P>
     * These can be used to get their associated Error List 
     */
    public Set getErrantDtos() {
        return getErrorKeys(true);
    }
    /**
     * Selects only errors keyed by a String and returns a Set of these Strings.  
     * These String are used to group related errors that do not map to a
     * specific DTO.  
     * <P>
     * These can be used to get their associated Error List 
     */
    public Set getOrphanErrorKeys() {
        return getErrorKeys(false);
    }
    
    private Set getErrorKeys(boolean selectDtos) {
        Iterator keys = getErrors().keySet().iterator();
        Set result = new HashSet(getErrors().size());
        while (keys.hasNext()) {
            Object key = keys.next();

            if (selectDtos && !(key instanceof String)) {
                result.add(key);
            }
            else if (!selectDtos && key instanceof String) {
                result.add(key);
            }
        }
        return result;
    }
    public List getErrorsForDto(Object dto) {
        return (List)getErrors().get(dto);
    }
    
    public void setInstitution(Institution institution) {
        Iterator dtos = getDtoIterator();
        while (dtos.hasNext()) {
            Object dto = dtos.next();
            if (dto instanceof HealthRecordDomainObject ) 
            { 
                ((HealthRecordDomainObject)dto).setInstitution(institution);
            }
        }
    }

	public DtoListPayload(int size) {
		super();
		setDtoList(new ArrayList(size));
	}
    public DtoListPayload( ) {
        super(); 
    }
    
	public List getDtoList() {
        if (dtoList == null) {
            dtoList = new ArrayList();
        }
        return dtoList;
    }
	
	protected void setDtoList(List aDtoList) {dtoList = aDtoList;}
   
	public Iterator getDtoIterator() {
		return getDtoList().iterator();
	}
    
	public void addDto(Object dto) {
        if (dto instanceof DtoListPayload) {
            addAll((DtoListPayload)dto);
        }
        else {
            getDtoList().add(dto);
        }
	}
    public boolean isEmpty() {
        return getDtoList().isEmpty();
    }
    public int size() {
        return getDtoList().size();
    }
    public boolean removeDto(Object dto) {
        return getDtoList().remove(dto);
    }
    
    public void addAll(DtoListPayload newList) {
        getDtoList().addAll(newList.getDtoList());
    }
    /**
     * Convenince method to give typed array for easy processing
     * 
     * @param array of size 0 of type required.  
     * @return the typed array ready to cast and use.  
     */
    public Object [] toArray(Object [] array) {
        return getDtoList().toArray(array);
    }
    
    public boolean containsErrors() {
        return getErrors().size() > 0;
    }
    /**
     * Adds the errors under the key (dto) provided.  If errors already exist
     * for the key, the new errors are apended to the existing errors. 
     * @param dto - intended to be the DTO to which the errors pertain,
     *              but could be anything and will be String for orphan errors 
     * @param errors
     */
    public void addErrorList(Object dto, List errors) {
        if (getErrors().containsKey(dto)) {
           ((List) getErrors().get(dto)).addAll(errors);
        }
        else {
            getErrors().put(dto, errors);
        }
    }
    
    public Map getErrors() {
        if (errors == null) {
            errors = new HashMap();
        }
        return errors;
    }
    public void setErrors(Map errors) {
        this.errors = errors;
    }
    public MessagingRequest getRequest() {
        return request;
    }
    public void setRequest(MessagingRequest request) {
        this.request = request;
    }
 }
